class Router {
    constructor(basePath = null) {
      this.routes = [];
      this.defaultRoute = null;

      if (basePath) {
        this.base = basePath.replace(/\/+$/, '') + '/';
      } else {
        const scripts = document.getElementsByTagName('script');
        const currentScript = scripts[scripts.length - 1].getAttribute('src') || '';
        const url = new URL(currentScript, window.location.href);

        const currentPath = window.location.pathname.split('/');
        const scriptPath = url.pathname.split('/');

        let baseParts = [];
        for (let i = 0; i < currentPath.length; i++) {
          if (currentPath[i] === scriptPath[i]) break;
          baseParts.push(currentPath[i]);
        }

        this.base = baseParts.join('/') + '/';
      }
    }

    addRoute(pattern, callback) {
      const fullPattern = this._normalizePath(pattern).split('?')[0];
      const paramNames = [];
      const regex = new RegExp(
        '^' + fullPattern
          .replace(/\/:([^\/]+)/g, (_, name) => {
            paramNames.push(name);
            return '/([^/]+)';
          }) + '$'
      );
      this.routes.push({ regex, callback, paramNames });
    }

    setDefault(callback) {
      this.defaultRoute = callback;
    }

    navigate(path) {
      const fullPath = this._normalizePath(path);
      history.pushState({}, '', fullPath);
      this._loadRoute(fullPath);
    }

    _normalizePath(path) {
      const url = new URL(path, window.location.origin + this.base);
      return this.base + url.pathname.replace(this.base, '').replace(/^\/+/, '') + url.search;
    }

    _getCurrentPath() {
      return window.location.pathname + window.location.search;
    }

    _parseQuery(queryString) {
      const params = {};
      const searchParams = new URLSearchParams(queryString);
      for (const [key, value] of searchParams.entries()) {
        params[key] = value;
      }
      return params;
    }

    _loadRoute(fullPath = null) {
      const url = new URL(fullPath || window.location.href, window.location.origin);
      const pathname = url.pathname;
      const queryParams = this._parseQuery(url.search);

      for (const route of this.routes) {
        const match = pathname.match(route.regex);
        if (match) {
          const params = { ...queryParams };
          route.paramNames.forEach((name, i) => {
            params[name] = match[i + 1];
          });
          route.callback(params);
          return;
        }
      }

      if (this.defaultRoute) {
        this.defaultRoute(pathname + url.search);
      }
    }

    curpath(){

        const currentPath = window.location.pathname;
        const base = this.base.replace(/\/+$/, '');
        const relativePath = currentPath.startsWith(base) ? currentPath.slice(base.length) : currentPath;
        return '/' + relativePath.replace(/^\/+/, '');

    }
    start() {
      window.addEventListener('popstate', () => {
        this._loadRoute();
      });
      this._loadRoute(this._getCurrentPath());
    }
};
